% This function calculates the terms in Eq.24, 27, 30 & 33.

% Sc (terms in Eq.24, 27, 30 & 33):
% 1)v1; 2)v2; 3)LTr; 4)var(Tr); 5)v3; 6)v4; 7)LTn; 8)var(Tn); 9)c;
% 10)LTrTn; 11)cov(Tr,Tn); 12)var(Tq).

% Ra: catchment-average rainfall excess;
% Rt: storm-average rainfall excess;
% Rat: catchment- and storm-average rainfall excess;
% psi: ratio of catchment- and storm-average rainfall excess;
% ET: expectation of runoff generation time;
% ThetaI/ThetaO: subsurface/surface routing time field;
% Xi: ratio between the psi-weighted average of alpha and alphai;
% ETheta: expectation of runoff routing time;
% CCI/CCO: spatial covariance between routing time and rainfall excess field;

function Sc=AFS(Ra,Rt,Rat,psi,ETr,ThetaI,ThetaO,Xi,ETn,CCI,CCO)

Area=find(~isnan(ThetaI));

% Equation 27 (25)
v1=[(size(Ra,1)^2+1)/12 (size(Ra,1)^2+1)/12 (size(Ra,1)^2+1)/12];

cc2=cov((1:size(Ra,1)).^2,Ra(:,1),1);
cc=cov(1:size(Ra,1),Ra(:,1),1);
vrI2=(cc2(2,1)-cc(2,1)*(size(Ra,1)+1)-cc(2,1)^2/Rat(1))/Rat(1);
cc2=cov((1:size(Ra,1)).^2,Ra(:,2),1);
cc=cov(1:size(Ra,1),Ra(:,2),1);
vrO2=(cc2(2,1)-cc(2,1)*(size(Ra,1)+1)-cc(2,1)^2/Rat(2))/Rat(2);

v2=[vrI2 vrO2 psi(1)*vrI2+psi(2)*vrO2];

LTr=psi(1)*(ETr(1)-ETr(3)).^2+psi(2)*(ETr(2)-ETr(3)).^2;

% Equation 30 (28)
v3=[var(ThetaI(Area),1) var(ThetaO(Area),1)...
    psi(1)*Xi(1)^2*var(ThetaI(Area),1)+psi(2)*Xi(2)^2*var(ThetaO(Area),1)];

rtI=Rt(:,:,1);
cc2=cov(ThetaI(Area).^2,rtI(Area),1);
cc=cov(ThetaI(Area),rtI(Area),1);
vrI4=(cc2(2,1)-2*mean(ThetaI(Area))*cc(2,1)-cc(2,1)^2/Rat(1))/Rat(1);
rtO=Rt(:,:,2);
cc2=cov(ThetaO(Area).^2,rtO(Area),1);
cc=cov(ThetaO(Area),rtO(Area),1);
vrO4=(cc2(2,1)-2*mean(ThetaO(Area))*cc(2,1)-cc(2,1)^2/Rat(2))/Rat(2);

v4=[vrI4 vrO4 psi(1)*Xi(1)^2*vrI4+psi(2)*Xi(2)^2*vrO4];

LTn=psi(1)*(Xi(1)*ETn(1)-ETn(3))^2+psi(2)*(Xi(2)*ETn(2)-ETn(3))^2;

% Equation 33 (31)
ccT=cov(1:size(Ra,1),Ra(:,1),1);
ccTheta=cov(ThetaI(Area),rtI(Area),1);
ccTTheta=cov(1:size(Ra,1),CCI,1);
cvI=ccTTheta(2,1)/Rat(1)-ccT(2,1)*ccTheta(2,1)/Rat(1)^2;
ccT=cov(1:size(Ra,1),Ra(:,2),1);
ccTheta=cov(ThetaO(Area),rtO(Area),1);
ccTTheta=cov(1:size(Ra,1),CCO,1);
cvO=ccTTheta(2,1)/Rat(2)-ccT(2,1)*ccTheta(2,1)/Rat(2)^2;

c=[cvI cvO psi(1)*Xi(1)*cvI+psi(2)*Xi(2)*cvO];

LTrTn=psi(1)*(Xi(1)*ETr(1)*ETn(1)-ETr(3)*ETn(3))+...
    psi(2)*(Xi(2)*ETr(2)*ETn(2)-ETr(3)*ETn(3));

% Equation 24
vrTr=nansum([v1;v2;[zeros(1,2) LTr]],1);
vrTn=nansum([v3;v4;[zeros(1,2) LTn]],1);
cvTrTn=c+[zeros(1,2) LTrTn];
vrTq=nansum([vrTr;vrTn;2*cvTrTn],1);

% Outputs (12 by 3)
Sc=[v1;v2;[0 0 LTr];vrTr;v3;v4;[0 0 LTn];vrTn;c;[0 0 LTrTn];cvTrTn;vrTq];
